<?php
/**
 * File Downloaded
 *
 * @package     AutomatorWP\Integrations\WP_Download_Manager\Triggers\File_Downloaded
 * @author      AutomatorWP <contact@automatorwp.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.0.0
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

class AutomatorWP_Download_Manager_File_Downloaded extends AutomatorWP_Integration_Trigger {

    public $integration = 'wp_download_manager';
    public $trigger = 'wp_download_manager_file_downloaded';

    /**
     * Register the trigger
     *
     * @since 1.0.0
     */
    public function register() {

        automatorwp_register_trigger( $this->trigger, array(
            'integration'       => $this->integration,
            'label'             => __( 'A file gets downloaded', 'automatorwp-pro' ),
            'select_option'     => __( 'A <strong>file</strong> gets downloaded', 'automatorwp-pro' ),
            /* translators: %1$s: Product title. %2$s: Number of times. */
            'edit_label'        => sprintf( __( 'File %1$s gets downloaded %2$s time(s)', 'automatorwp-pro' ), '{post}', '{times}' ),
            /* translators: %1$s: Product title. */
            'log_label'         => sprintf( __( 'File %1$s gets downloaded', 'automatorwp-pro' ), '{post}' ),
            'action'            => 'wpdm_onstart_download',
            'function'          => array( $this, 'listener' ),
            'priority'          => 10,
            'accepted_args'     => 1,
            'options'           => array(
                'post' => automatorwp_utilities_post_option( array(
                    'name' => __( 'Download:', 'automatorwp-pro' ),
                    'option_none_label' => __( 'any download', 'automatorwp-pro' ),
                    'post_type' => 'wpdmpro'
                ) ),
                'times' => automatorwp_utilities_times_option(),
            ),
            'tags' => array_merge(
                automatorwp_download_manager_download_tags(),
                automatorwp_utilities_times_tag()
            )
        ) );

    }

    /**
     * Trigger listener
     *
     * @since 1.0.0
     *
     * @param string    $package
     * 
     */
    public function listener( $package ) {

        // Bail if file is not publish
        if ( $package['post_status'] !== 'publish' && $package['post_status'] !== 'private' ) {
            return;
        }
        
        $user_id = get_current_user_id();

        // Bail if user is not logged
        if ($user_id === 0) {
            return;
        }

        // Trigger user order completed
        automatorwp_trigger_event( array(
            'trigger'       => $this->trigger,
            'user_id'       => $user_id,
            'post_id'       => $package['ID'],
            'download_id'   => $package['ID'],
            'download_title' => $package['title'],
            'download_description' => $package['description'],
            'download_excerpt' => $package['excerpt'],
        ) );
     
    }

    /**
     * User deserves check
     *
     * @since 1.0.0
     *
     * @param bool      $deserves_trigger   True if user deserves trigger, false otherwise
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return bool                          True if user deserves trigger, false otherwise
     */


     public function user_deserves_trigger( $deserves_trigger, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Don't deserve if post is not received
        if( ! isset( $event['post_id'] ) ) {
            return false;
        }

        // Don't deserve if post doesn't match with the trigger option
        if( ! automatorwp_posts_matches( $event['post_id'], $trigger_options['post'] ) ) {
            return false;
        }

        return $deserves_trigger;

    }

    /**
     * Register the required hooks
     *
     * @since 1.0.0
     */

    public function hooks() {

        // Log meta data
        add_filter( 'automatorwp_user_completed_trigger_log_meta', array( $this, 'log_meta' ), 10, 6 );

        parent::hooks();
    }

    /**
     * Trigger custom log meta
     *
     * @since 1.0.0
     *
     * @param array     $log_meta           Log meta data
     * @param stdClass  $trigger            The trigger object
     * @param int       $user_id            The user ID
     * @param array     $event              Event information
     * @param array     $trigger_options    The trigger's stored options
     * @param stdClass  $automation         The trigger's automation object
     *
     * @return array
     */

    function log_meta( $log_meta, $trigger, $user_id, $event, $trigger_options, $automation ) {

        // Bail if action type don't match this action
        if( $trigger->type !== $this->trigger ) {
            return $log_meta;
        }

        $log_meta['download_id'] = ( isset( $event['download_id'] ) ? $event['download_id'] : 0 );
        $log_meta['download_title'] = ( isset( $event['download_title'] ) ? $event['download_title'] : '' );
        $log_meta['download_description'] = ( isset( $event['download_description'] ) ? $event['download_description'] : '' );
        $log_meta['download_excerpt'] = ( isset( $event['download_excerpt'] ) ? $event['download_excerpt'] : '' );

        return $log_meta;

    }

}


new AutomatorWP_Download_Manager_File_Downloaded();